package miniDI.framework;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import miniDI.framework.annotations.Pair;
import miniDI.framework.annotations.Pairs;
import miniDI.framework.annotations.ServiceDeclarationPackage;
import miniDI.framework.annotations.ServiceImplementationPackage;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

@ServiceDeclarationPackage
// This class keeps  alla mappings between interfaces and implementations
// Has a method that, given an interface,  allows retrieving the relative implementation.
// Subclasses need only to contain the needed annotation
public abstract class AbstractConfigurator {

    // this HashMap keeps the mappings between interfaces and implementations
    private Map<Class<?>, Class<?>> classMap = new HashMap<>();

    // this method extracts the info from annotations, and saves them in the Map
    protected AbstractConfigurator() {
        File xml = new File("miniDI.xml");
        System.out.println("looking for file " + xml.getAbsolutePath() + "...");

        if (xml.exists()) {
            System.out.println("...found. Proceeding with XML configuration");
            configureWithXML(xml);
        } else {
            System.out.println("...not found. Proceeding with annotation configuration");
            configureWithAnnotations();
        }
    }

    private void configureWithAnnotations() {
        // From the annotation find out which package hosts the service declarations
        ServiceDeclarationPackage sd = (ServiceDeclarationPackage) this.getClass().getAnnotation(ServiceDeclarationPackage.class);
        // From the annotation find out which package hosts the service implementation
        ServiceImplementationPackage si = (ServiceImplementationPackage) this.getClass().getAnnotation(ServiceImplementationPackage.class);
        // From the annotation, get all the pairs declaration-imlementation, 
        // and create a mapping between the components of each pair        
        Pairs p = (Pairs) this.getClass().getAnnotation(Pairs.class);
        Pair[] coppie = p.value();
        for (Pair c : coppie) {
            insertDataIntoMap(sd.value(),c.key(),si.value(),c.value() );
        }
    }

    private void configureWithXML(File xmlFile) {
        DocumentBuilderFactory domFactory =
                DocumentBuilderFactory.newInstance();
        domFactory.setNamespaceAware(true);
        Document doc = null;
        try {
            DocumentBuilder builder = domFactory.newDocumentBuilder();
            doc = builder.parse(xmlFile.getAbsolutePath());
        } catch (ParserConfigurationException | SAXException | IOException ex) {
            Logger.getLogger(AbstractConfigurator.class.getName()).log(Level.SEVERE, null, ex);
        }
        // prepare the XPath expression 
        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();
        String serviceDeclarationPackage = null, serviceImplementationPackage = null;
        NodeList keys = null, values = null;
        try {
            XPathExpression expr1 = xpath.compile("/miniDI/ServiceDeclarationPackage/text()");
            serviceDeclarationPackage = ((Node) expr1.evaluate(doc, XPathConstants.NODE)).getNodeValue().trim();
            //
            expr1 = xpath.compile("/miniDI/ServiceImplementationPackage/text()");
            serviceImplementationPackage = ((Node) expr1.evaluate(doc, XPathConstants.NODE)).getNodeValue().trim();
            // get the set of keys
            expr1 = xpath.compile("/miniDI/Pairs/Pair/key/text()");
            keys = (NodeList) expr1.evaluate(doc, XPathConstants.NODESET);
            // get the set of values
            expr1 = xpath.compile("/miniDI/Pairs/Pair/value/text()");
            values = (NodeList) expr1.evaluate(doc, XPathConstants.NODESET);
        } catch (XPathExpressionException ex) {
            Logger.getLogger(AbstractConfigurator.class.getName()).log(Level.SEVERE, null, ex);
        }
        // From the xml, get all the pairs declaration-imlementation, 
        // and create a mapping between the components of each pair
        for (int i = 0; i < keys.getLength(); i++) {
            insertDataIntoMap(
                    serviceDeclarationPackage,keys.item(i).getNodeValue().trim(),
                    serviceImplementationPackage,values.item(i).getNodeValue().trim() 
                    );
        }
    }

    private void insertDataIntoMap(
            String serviceDeclarationPackage,
            String key,
            String serviceImplementationPackage,
            String value) {
        Class keyClass = null, valueClass = null;
        try {
            keyClass = Class.forName(serviceDeclarationPackage + "." + key);
            valueClass = Class.forName(serviceImplementationPackage + "." + value);
            System.out.println("Found mapping:\n"+
                    "==>"+serviceDeclarationPackage + "." + key+
                    "==>"+serviceImplementationPackage + "." + value);
        } catch (ClassNotFoundException ex) {
            Logger.getLogger(AbstractConfigurator.class.getName()).log(Level.SEVERE, null, ex);
        }
        createMapping(keyClass, valueClass);
    }
    // record a mapping in the Map
    private <T> void createMapping(Class<T> baseClass, Class<? extends T> subClass) {
        classMap.put(baseClass, subClass.asSubclass(baseClass));
    }

    // find which implementation (return value) corresponds to a given interface (input parameter)
    <T> Class<? extends T> getMapping(Class<T> type) {
        Class<?> implementation = classMap.get(type);

        if (implementation == null) {
            throw new IllegalArgumentException("Couldn't find the mapping for : " + type);
        }

        return (Class<T>) implementation;// .asSubclass(type);
    }
}